<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\UsersImport;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use App\Models\Nominees;
use App\Models\Payment;
use BinshopsBlog\Models\BinshopsPostTranslation;
use Carbon\Carbon;
use Illuminate\Support\Facades\Session;


class UserController extends Controller
{
    public function showUploadForm()
    {
        return view('upload');
    }

    public function upload(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xls,xlsx|max:2048',
        ]);

        // Increase the maximum execution time to handle large files
        ini_set('max_execution_time', 300); // 5 minutes

        Excel::import(new UsersImport, $request->file('file'));

        return redirect()->route('home')->with('success', 'Users imported successfully!');
    }

    public function edit()
    {
        return view('user.edit');
    }

    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $user->id,
            'password' => 'nullable|string|min:8|confirmed',
            'bio' => 'nullable|string|max:255',
            'gender' => 'nullable|in:Male,Female,Other',
            'cug_no' => 'nullable|string|max:255',
            'designation' => 'nullable|string|max:255',
            'posting_area' => 'nullable|string|max:255',
            'home_town' => 'nullable|string|max:255',
            'date_of_birth' => 'nullable|date',
            'marriage_anniversary' => 'nullable|date',
            'state' => 'nullable|string|max:255',
            'zone' => 'nullable|string|max:255',
            'contact_privacy' => 'nullable|boolean',
            'profile_photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'cover_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $data = $request->except(['_token', '_method', 'password_confirmation']);

        if ($request->has('password')) {
            $data['password'] = Hash::make($request->password);
        }

        $user->update($data);

        // Handle profile photo upload
        if ($request->hasFile('profile_photo')) {
            $profilePhoto = $request->file('profile_photo');
            $profilePhotoName = time() . '_' . $profilePhoto->getClientOriginalName();
            $profilePhoto->move(public_path('user/profile_photo'), $profilePhotoName);
            $user->profile_photo = $profilePhotoName;
        }

        // Handle cover image upload
        if ($request->hasFile('cover_image')) {
            $coverImage = $request->file('cover_image');
            $coverImageName = time() . '_' . $coverImage->getClientOriginalName();
            $coverImage->move(public_path('user/cover_image'), $coverImageName);
            $user->cover_image = $coverImageName;
        }

        $user->save();

        return redirect()->back()->with('success', 'Profile updated successfully.');
    }

    public function list()
    {
        $users = User::all();
        $posting_area =  $designation = '';
        $cities = User::distinct()->pluck('posting_area');
        $designations = User::distinct()->pluck('designation');

        return view('user.list', compact('users', 'cities', 'designations', 'designation', 'posting_area'));
    }

    public function sort(Request $request)
    {
        $users = User::query();
        $posting_area =  $designation = '';

        if ($request->filled('posting_area') && $request->filled('designation')) {
            $users->where('posting_area', $request->posting_area);
            $users->where('designation', $request->designation);
            $posting_area = $request->posting_area;
            $designation = $request->designation;
        } elseif ($request->filled('posting_area')) {
            $users->where('posting_area', $request->posting_area);
            $posting_area = $request->posting_area;
        } elseif ($request->filled('designation')) {
            $users->where('designation', $request->designation);
            $designation = $request->designation;
        }

        $users = $users->get();

        return response()->json([
            'users' => $users,
            'posting_area' => $posting_area,
            'designation' => $designation
        ]);
    }

    public function uploadImage(Request $request)
    {
        try {
            $request->validate([
                'image' => 'required|image|mimes:jpeg,png,jpg,gif|max:5120',
            ], [
                'image.max' => 'The image size too large. Either Zoom in the image or use another image', // Custom error message for image size
            ]);

            $profilePhoto = $request->file('image');
            $profilePhotoName = time() . '_' . $profilePhoto->getClientOriginalName();
            $profilePhoto->move(public_path('user/profile_photo'), $profilePhotoName);

            // Here you can update the user's profile_photo field assuming you have a User model
            // Replace User with your actual model name
            $user = auth()->user(); // Assuming you have user authentication
            $user->profile_photo = $profilePhotoName;
            $user->save();

            return response()->json(['success' => 'You have successfully uploaded the image.', 'image' => $profilePhotoName], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to upload image. ' . $e->getMessage()], 500);
        }
    }

    public function uploadImagecover(Request $request)
    {
        // Validation with custom error messages
        $request->validate([
            'image' => 'required|image|mimes:jpeg,png,jpg,gif|max:5120',
        ], [
            'image.max' => 'The image size too large. Either Zoom in the image or use another image', // Custom error message for image size
        ]);

        try {
            $profilePhoto = $request->file('image');
            $profilePhotoName = time() . '_' . $profilePhoto->getClientOriginalName();
            $profilePhoto->move(public_path('user/cover_image'), $profilePhotoName);

            // Here you can update the user's cover_image field assuming you have a User model
            $user = auth()->user(); // Assuming you have user authentication
            $user->cover_image = $profilePhotoName;
            $user->save();

            return response()->json(['success' => 'You have successfully uploaded the image.', 'image' => $profilePhotoName], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Failed to upload image. ' . $e->getMessage()], 500);
        }
    }

    public function details($id)
    {
        $user = User::findOrFail($id);
        $posts = array();
        $posts = BinshopsPostTranslation::join('binshops_posts', 'binshops_post_translations.post_id', '=', 'binshops_posts.id')
            ->where("binshops_posts.is_published", true)
            ->where("binshops_posts.user_id", $id) // specify table name or alias for user_id
            ->orderBy("binshops_post_translations.created_at", "desc") // specify table name or alias for posted_at
            ->paginate(config("binshopsblog.per_page", 10));
        foreach ($posts as $post) {
            $post->image_url = $this->extractImageUrl($post);

            // Fetch user data based on user_id
            $user = User::find($id);
            // Add user data to the post object
            $post->user = $user;
        }
        return view('user.details', compact('user', 'posts'));
    }
    private function extractImageUrl($post)
    {
        // Generate the HTML for the image tag
        $image_html = $post->image_tag("large", true, '');

        // Initialize the image URL variable
        $image_url = '';

        if (!empty($image_html)) {
            // Load the HTML into DOMDocument
            $doc = new \DOMDocument();
            libxml_use_internal_errors(true); // To suppress warnings due to malformed HTML
            $doc->loadHTML($image_html);
            libxml_clear_errors();

            // Extract the image URL from the DOM
            $img_tags = $doc->getElementsByTagName('img');
            if ($img_tags->length > 0) {
                $image_url = $img_tags->item(0)->getAttribute('src');
            }
        }

        return $image_url;
    }

    public function updateBio(Request $request)
    {

        $user = auth()->user(); // Assuming the user is authenticated
        $user->bio = $request->input('bio');
        $user->save();

        return response()->json(['success' => true, 'message' => 'Bio updated successfully']);
    }
    public function update_p_info(Request $request)
    {
        $user = auth()->user();
        $data = $request->data;
        $user->name = $data['employee_name'];
        $user->fathers_name = $data['fathers_name'];
        $user->gender = $data['sex'];
        $user->date_of_birth = $data['dob'];
        $user->email = $data['email'];
        $user->home_district = $data['home_district'];
        $user->home_mandal = $data['home_mondal'];
        $user->religion = $data['religion'];
        $user->retirement_year = $data['retirement_date'];
        $user->contact = $data['mobile_no'];
        $user->category = $data['category'];
        $save = $user->save();
        return response()->json(['success' => true, 'message' => 'Personal Information updated successfully']);
    }

    public function updatePostingInfo(Request $request)
    {
        $user = Auth::user();

        $validatedData = $request->validate([
            'designation' => 'required|string|max:255',
            'current_office' => 'required|string|max:255',
            'present_posting_joining_date' => 'required|date',
            'zone' => 'required|string|max:255',
            'batch' => 'required|string|ma  x:255',
            'current_mandal' => 'required|string|max:255',
        ]);

        $user->update($validatedData);

        return response()->json(['success' => true]);
    }

    public function updatePassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'password' => 'required|string|min:6|confirmed',
            'password_confirmation' => 'required|same:password',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 400);
        }

        $user = auth()->user();
        $user->password = Hash::make($request->password);
        $user->save();
        session(['step' => 'completed']);

        return response()->json(['success' => true], 200);
    }


    public function updatePasswordajax(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'password' => 'required|string|min:6|confirmed',
            'password_confirmation' => 'required|same:password',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 400);
        }

        $user = auth()->user();
        $user->password = Hash::make($request->password);
        $user->registration_type = null ;
        $user->save();

        return response()->json(['success' => true], 200);
    }

    public function offline_annual(Request $request)
    {
        return view('user.offline_pay_annual');
    }

    public function offline_benevolent(Request $request)
    {
        return view('user.offline_pay_benevolent');
    }

    public function update_jd(Request $request)
    {

        // Assuming you have a model to handle the data
        $jurisdiction = auth()->user(); // or the appropriate ID
        $jurisdiction->joining_office = $request->input('joiningOffice');
        $jurisdiction->joining_year = $request->input('joiningDate');
        $jurisdiction->order_no = $request->input('orderNo');
        $jurisdiction->order_date = $request->input('orderDate');
        $jurisdiction->save();

        return response()->json(['message' => 'Data updated successfully!']);
    }

    public function updae_nominee_index($id)
    {
        return view('user.update_nominee', compact('id'));
    }

    public function save_nominee(Request $request)
    {
        $nominee = Nominees::create([
            'name' => $request->input('name'),
            'date_of_birth' => $request->input('birthday'),
            'gender' => $request->input('gender'),
            'relationship' => $request->input('relationship'),
            'payment_id' => $request->input('payment_id'),
        ]);

        return redirect()->route('b_fund');
    }

    public function getEmailByUserId(Request $request)
    {
        $employeeId = $request->input('employee_id');
        $user = User::where('employee_id', $employeeId)->first();
		if ($user) {
            return response()->json(['email' => $user->email]);
        } else {
            return response()->json(['error' => 'User not found'], 404);
        }
    }

    public function validateEmail(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email|unique:users,email'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first('email')
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Email is valid and unique'
        ]);
    }

    public function updateEmail(Request $request)
    {
        $user = Auth::user();
        $user->email = $request->email;
        $user->save();
        session(['step' => 'password']);
        return response()->json(['success' => true, 'message' => 'Email updated successfully']);
    }
    public function findUserByEmployeeId(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|string',
        ]);

        $employeeId = $request->input('employee_id');
        $user = User::where('employee_id', $employeeId)->first();

        if ($user) {
            $payments = $user->payments()->where('package_name', 'Annual Subscription')->get()->first();

            if(auth()->user()->role === 'admin'){
                return view('admin.benevolentfund', ['user' => $user, 'employee_id' => $employeeId ,'payments' => $payments]);
            } else if(auth()->user()->role === 'sub-admin'){
                return view('sub-admin.benevolentfund', ['user' => $user, 'employee_id' => $employeeId, 'payments' => $payments]);
            }
        } else {

            return redirect()->back()->with('error', 'No user found with ID:'. $employeeId );
        }
    }

    public function findUser(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|string',
        ]);

        $employeeId = $request->input('employee_id');
        $user = User::where('employee_id', $employeeId)->first();

        if ($user) {
            $payments = $user->b_fund_payments()->get()->first();

            if(auth()->user()->role === 'admin'){
                return view('admin.benevolentfund', ['user' => $user, 'employee_id' => $employeeId ,'payments' => $payments]);
            } else if(auth()->user()->role === 'sub-admin'){
                return view('sub-admin.benevolentfund_b', ['user' => $user, 'employee_id' => $employeeId, 'payments' => $payments]);
            }
        } else {

            return redirect()->back()->with('error', 'No user found with ID:'. $employeeId );
        }
    }

	 public function pan_index()
    {
        return view('user.pan');
    }
 public function society_registration_index()
    {
        return view('user.society_registration');
    }
 public function memorandum_index()
    {
        return view('user.memorandum_of_association');
    }
 public function asssociation_index()
    {
        return view('user.rules_of_association');
    }
 public function terms_index()
    {
        return view('user.terms');
    }
 public function privacy_policy_index()
    {
        return view('user.privacy_policy');
    }

 public function view_receipt(Request $request)
    {

        $employeeId = $request->input('employee_id');
        $user = User::where('employee_id', $employeeId)->first();
        $details = Payment::where('package_name', 'Annual Subscription')
            ->where('user_id', $user->id)
            ->where('payment_status', 'success')
            ->with('cashier') // Eager load the cashier relationship
            ->first();

        $cashierName = $details && $details->cashier ? $details->cashier->name : 'No cashier assigned';
        return view('user.receipt', compact('details', 'cashierName'));
    }
}
