<?php

namespace App\Http\Controllers;

use App\Models\BFundPayment;
use App\Models\Nominees;
use App\Models\Payment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\HttpFoundation\StreamedResponse;
use PDF;

class SubscriptionController extends Controller
{
    public function index()
    {
        $user = auth()->user();
        $user_status = $user->annual_status;
        if ($user->annual_status === 1) {
            $details = Payment::where('package_name', 'Annual subscription')
                ->where('user_id', $user->id)
                ->where('payment_status', 'success')
                ->first();
        } else if ($user->annual_status === 2) {
            $details = Payment::where('package_name', 'Annual subscription')
                ->where('user_id', $user->id)
                ->where('payment_status', 'pending approval')
                ->first();
        } else {
            $details = '';
        }
        return view('user.subscription', compact('details', 'user_status'));
    }

    public function b_fund_index()
    {
        $user = auth()->user();
        $user_status = $user->b_fund_status;
        $details = null;
        $hide = false;

        if ($user_status) {
            $details = BFundPayment::where('package_name', 'Benevolent Fund')
                ->where('user_id', $user->id)
                ->first();

            if ($details) {
                $check = Nominees::where('payment_id', $details->id)->exists();
                $hide = $check;
            }
        }

        return view('user.b_fund', compact('details', 'user_status', 'user', 'hide'));
    }


    public function payments_index()
    {
        return view('user.payments');
    }

    public function submitOfflineSubscription(Request $request)
{
    $user = auth()->user();
    $user_id = $user->id;
    $packageDetails = $this->getPackageDetails($request->type);

    // Update user status
    $user->{$packageDetails['status_field']} = 2;

    // Handle file upload
    $receiptDocumentPath = $this->handleFileUpload($request);

    // Create payment record
    $this->createPaymentRecord($request, $user_id, $packageDetails, $receiptDocumentPath);

    // Save user
    $user->save();

    // Redirect with success message
    return redirect()->route($packageDetails['route'])->with('success', 'Subscription submitted successfully!');
}

private function getPackageDetails($type)
{
    if ($type === 'Benevolent Fund') {
        return [
            'status_field' => 'b_fund_status',
            'package_name' => 'Benevolent Fund',
            'amount' => '20000',
            'id_field' => 'cheque_number',
            'route' => 'b_fund'
        ];
    } else {
        return [
            'status_field' => 'annual_status',
            'package_name' => 'Annual Subscription',
            'amount' => '2000',
            'id_field' => 'receipt_number',
            'route' => 'subscription'
        ];
    }
}

private function handleFileUpload($request)
{
    if ($request->hasFile('receipt_document')) {
        $file = $request->file('receipt_document');
        $destinationPath = 'receipts';
        $filename = time() . '_' . $file->getClientOriginalName();
        $file->move(public_path($destinationPath), $filename);
        return $destinationPath . '/' . $filename;
    }
    return null;
}

private function createPaymentRecord($request, $user_id, $packageDetails, $receiptDocumentPath)
{
    $paymentData = [
        'user_id' => $user_id,
        'package_name' => $packageDetails['package_name'],
        'amount' => $packageDetails['amount'],
        'payment_type' => 'offline',
        'doc_date' => $request->doc_date,
        $packageDetails['id_field'] => $request->{$packageDetails['id_field']},
        'receipt_document' => $receiptDocumentPath,
        'payment_status' => 'pending approval',
    ];

    if ($packageDetails['package_name'] === 'Benevolent Fund') {
        BFundPayment::create($paymentData);
    } else {
        Payment::create($paymentData);
    }
}


    public function export($type, $format = 'csv')
    {
        $user = auth()->user();

        if ($type === 'annual') {
            $details = payment::where('package_name', 'Annual subscription')
                ->where('user_id', $user->id)
                ->first();
        } else if ($type === 'b_fund') {
            $details = payment::where('package_name', 'Benevolent Fund')
                ->where('user_id', $user->id)
                ->first();
        } else {
            return redirect()->back()->with('error', 'No subscription details available.');
        }

        if ($format === 'pdf') {
            $pdf = PDF::loadView('subscription_pdf', compact('user', 'details'));
            $filename = 'invoice_' . $user->id . '_' . now()->format('Ymd_His') . '.pdf';
            return $pdf->download($filename);
        } else {
            $filename = 'invoice_' . $user->id . '_' . now()->format('Ymd_His') . '.csv';
            $response = new StreamedResponse(function () use ($user, $details) {
                $handle = fopen('php://output', 'w');

                // Add a prominent heading for Subscription Details
                fputcsv($handle, ['*** SUBSCRIPTION DETAILS ***']);
                fputcsv($handle, []); // Empty row for spacing

                // Add user details
                fputcsv($handle, ['USER NAME', $user->name]);
                fputcsv($handle, ['USER EMAIL', $user->email]);
                fputcsv($handle, ['USER CONTACT', $user->contact]);
                fputcsv($handle, []); // Empty row for spacing

                // Add a prominent heading for Subscription Details
                fputcsv($handle, ['*** SUBSCRIPTION DETAILS ***']);

                // Add subscription details
                fputcsv($handle, ['PACKAGE NAME', $details->package_name]);
                fputcsv($handle, ['PAYMENT TYPE', $details->payment_type]);
                fputcsv($handle, ['AMOUNT', $details->amount]);
                fputcsv($handle, ['TOKEN', $details->token]);
                fputcsv($handle, ['RAZORPAY ID', $details->razorpay_id]);
                fputcsv($handle, ['CREATED AT', $details->created_at]);

                fclose($handle);
            });

            $response->headers->set('Content-Type', 'text/csv');
            $response->headers->set('Content-Disposition', 'attachment; filename="' . $filename . '"');

            return $response;
        }
    }
    public function update(Request $request)
    {
        $request->validate([
            'payment_id' => 'required|integer|exists:payments,id',
            'receipt_number' => 'required|string|max:255',
            'receipt_document' => 'nullable|mimes:jpeg,png,jpg,gif,svg,pdf|max:2048',
        ]);

        $payment = Payment::find($request->payment_id);

        if (!$payment) {
            return redirect()->back()->with('error', 'Payment record not found.');
        }

        // Update receipt number
        $payment->receipt_number = $request->receipt_number;
        // Handle file upload if a new file is provided
        if ($request->hasFile('receipt_document')) {
            $file = $request->file('receipt_document');
            $destinationPath = 'receipts';
            $filename = time() . '_' . $file->getClientOriginalName();
            $file->move(public_path($destinationPath), $filename);
            $receiptDocumentPath = $destinationPath . '/' . $filename;

            if ($payment->receipt_document) {
                Storage::disk('public')->delete($payment->receipt_document);
                $payment->receipt_document = $receiptDocumentPath;
            }
        }
        $payment->save();

        return redirect()->back()->with('success', 'Subscription details updated successfully!');
    }
}
