<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use App\Mail\RegistrationSuccess;
use Illuminate\Support\Facades\Mail;
use Symfony\Component\HttpFoundation\Request;

class RegisterController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */

    use RegistersUsers;

    /**
     * Where to redirect users after registration.
     *
     * @var string
     */
    protected $redirectTo = '/registration_status';

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest');
    }

    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array  $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function validator(array $data)
    {
        return Validator::make($data, [
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'employee_id' => ['required',  'max:255', 'unique:users'],
            'contact' => ['required',  'max:255', 'unique:users'],
            'designation' => ['required', 'string'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
            'current_city' => ['required','string'],
            'terms' => ['required']
        ]);
    }

    public function validateRegister(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'employee_id' => ['required',  'max:255', 'unique:users'],
            'contact' => ['required',  'max:255', 'unique:users'],
            'designation' => ['required', 'string'],
            'password' => ['required', 'string', 'min:8'],
            'password_confirmation' => ['required','same:password'],
            'current_city' => ['required', 'string'],
            'terms' => ['required']
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()]);
        }

        return response()->json(['success' => 'Validation passed']);
    }


    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return \App\Models\User
     */
    protected function create(array $data)
    {
        $user = User::create([
            'name' => $data['name'],
            'email' => $data['email'],
            'employee_id' => $data['employee_id'],
            'contact'=> $data['contact'],
            'designation'=>$data['designation'],
            'password' => Hash::make($data['password']),
            'registration_status' => '2',  //under approval
            'registration_type' => null,
            'posting_area' => $data['current_city']
        ]);

        // Send the registration success email
        Mail::to($user->email)->send(new RegistrationSuccess($user));

        return $user;
    }
}
